//
// Created by 76538 on 3/3/2021.
//

#include "nanovoid_app.h"
#include "cxxopts.hpp"

#include <cstring>
#include <iostream>
#include <ctime>

#define PNG_BYTES_TO_CHECK 4

using namespace std;

class Args {
public:
  uint nsteps;
  string input;
  string output;
  string bucket_output;
  uint lshL, lshK; // Nx, Ny;
  double lshr;
};

Args* parse_args(int argc, const char* argv[]) {
  try
  {
    Args* args = new Args;
    
    cxxopts::Options options(argv[0], " - test forward simulation of grain growth.");
    options
      .positional_help("[optional args]")
      .show_positional_help();

    options
      .set_width(70)
      .set_tab_expansion()
      .allow_unrecognised_options()
      .add_options()
      ("s,nsteps", "Number of steps of simulation (default=100)", cxxopts::value<int>(), "N")
      ("o,output", "Output file (default=grain.out)", cxxopts::value<std::string>(), "FILE")
      ("i,input", "Input file (default=grain.in)", cxxopts::value<std::string>(), "FILE")
      ("lshK", "K for LSH (default=1)", cxxopts::value<int>(), "INT")
      ("lshL", "L for LSH (default=1)", cxxopts::value<int>(), "INT")
      ("lshr", "r for LSH (default=1e-4)", cxxopts::value<float>(), "FLOAT")
      ("bucket_output", "Output file of the bucket information (default=bucket.out)", cxxopts::value<std::string>(), "FILE")
      ("h,help", "Print help")
    #ifdef CXXOPTS_USE_UNICODE
      ("unicode", u8"A help option with non-ascii: à. Here the size of the"
        " string should be correct")
    #endif
    ;
      //("Nx", "size of x-axis (default=64)", cxxopts::value<int>(), "INT")
      //("Ny", "size of y-axis (default=64)", cxxopts::value<int>(), "INT")

    auto result = options.parse(argc, argv);

    if (result.count("help"))
    {
      std::cout << options.help({"", "Group"}) << std::endl;
      exit(0);
    }

    std::cout << "[Parse Args]" << std::endl;
    
    if (result.count("nsteps"))
    {
      std::cout << "  nsteps = " << result["nsteps"].as<int>() << std::endl;
      args->nsteps = (uint)result["nsteps"].as<int>();
    }else{
      args->nsteps = 501;
    }
    
    if (result.count("output"))
    {
      std::cout << "  output = " << result["output"].as<std::string>()
        << std::endl;
      args->output = result["output"].as<std::string>();
    }else{
      args->output = "nanovoid.out";
    }
    
    if (result.count("input"))
    {
      std::cout << "  input = " << result["input"].as<std::string>()
        << std::endl;
      args->input = result["input"].as<std::string>();
    }else{
      args->input = "nanovoid.in";
    }
    
    if (result.count("bucket_output"))
    {
      std::cout << "  bucket_output = " << result["bucket_output"].as<std::string>()
        << std::endl;
      args->bucket_output = result["bucket_output"].as<std::string>();
    }else{
      args->bucket_output = "bucket.out";
    }
    if (result.count("lshK"))
    {
      std::cout << "  lshK = " << result["lshK"].as<int>()
        << std::endl;
      args->lshK = (uint)result["lshK"].as<int>();
    }else{
      args->lshK = 1;
    }

    if (result.count("lshL"))
    {
      std::cout << "  lshL = " << result["lshL"].as<int>()
        << std::endl;
      args->lshL = (uint)result["lshL"].as<int>();
    }else{
      args->lshL = 1;
    }

    if (result.count("lshr"))
    {
      std::cout << "  lshr = " << result["lshr"].as<float>()
        << std::endl;
      args->lshr = (double)result["lshr"].as<float>();
    }else{
      args->lshr = 1e-4;
    }
    
    auto arguments = result.arguments();
    std::cout << "  Saw " << arguments.size() << " arguments" << std::endl;

    std::cout << "[End of Parse Args]" << std::endl;

    /*
    if (result.count("Nx"))
    {
      std::cout << "  Nx = " << result["Nx"].as<int>()
        << std::endl;
      args->Nx = (uint)result["Nx"].as<int>();
    }else{
      args->Nx = 64;
    }
    if (result.count("Ny"))
    {
      std::cout << "  Ny = " << result["Ny"].as<int>()
        << std::endl;
      args->Ny = (uint)result["Ny"].as<int>();
    }else{
      args->Ny = 64;
    }
    */
    
    return args;
  }
  catch (const cxxopts::OptionException& e)
  {
    std::cout << "error parsing options: " << e.what() << std::endl;
    exit(1);
  }
}



int main(int argc, const char* argv[]) {
  Args* args = parse_args(argc, argv);
  
    uint Nx = 256;
    uint Ny = 256;
    uint channel = 3;

    ParameterSet para(0.0); // = new ParameterSet();
    para.energy_v0 = 8.;
    para.energy_i0 = 8.;
    para.kBT0 = 5.;
    para.kappa_v0 = 1.0;
    para.kappa_i0 = 1.0;
    para.kappa_eta0 = 1.0;
    para.r_bulk0 = 5.0;
    para.r_surf0 = 10.0;
    para.p_casc0 = 0.01;
    para.bias0 = 0.3;
    para.vg0 = 0.01;
    para.diff_v0 = 1.0;
    para.diff_i0 = 1.0;
    para.L0 = 1.0;


    uint nsteps = args->nsteps; //501;

    valueType lshr = args->lshr; //0.5;
    uint lshK = args->lshK; //10;
    uint lshL = args->lshL; //1;


    std::string init_file = "../nanovoid_init_multi1_N256_c6_r10_std1.txt";
    valueType*** org_mat = read_from_data(init_file.c_str(), Nx, Ny);

    NanoVoidOneStep one_step(Nx, Ny, para, lshr, lshK, lshL);
    printf("before encode_from_img\n");

    struct timespec tstart={0,0}, tend={0,0};
    clock_gettime(CLOCK_MONOTONIC, &tstart);
    
    one_step.encode_from_img(org_mat);
    printf("finish encode_from_img\n");
    delete_3d_array(org_mat, Nx, Ny, 3);
    //one_step.inv.check_from_dfslist(one_step.num_items);

    for (int step = 0; step < nsteps; step++) {
        one_step.next();

        if (step % 10 == 0) {
          printf("step=%u, ", step);
          one_step.hash_t.clean_up_l_list();

          clock_gettime(CLOCK_MONOTONIC, &tend);
          printf("time_elapsed=%lf\n", ((double)tend.tv_sec + 1.0e-9*tend.tv_nsec) -
                 ((double)tstart.tv_sec + 1.0e-9*tstart.tv_nsec));
        }
        
        
        if (step % 1 == 0 || step == 1) {
            valueType*** decoded_img = one_step.decode_to_img();
            int return_code = 0;
            //string filename = "./output/nanovoid_cpp_multi1_N1024_c10_r50_std20/write_back_" + to_string(step) + ".txt";
            //string filename = "./output/nanovoid_cpp_multi1_N1024_c20_r25_std2/write_back_" + to_string(step) + ".txt";
            string filename = "../nn_forward_2/sim_" + to_string(step) + ".txt";
            return_code = writeData(filename.c_str(), Nx, Ny, decoded_img);
            if (return_code == 1) {
                printf("write back has error\n");
            }
            delete_3d_array(decoded_img, Nx, Ny, 3);
            one_step.inv.check_from_dfslist(one_step.num_items);
            printf("step=%d\n", step);
        }
        
    }

    clock_gettime(CLOCK_MONOTONIC, &tend);

    delete args;

    printf("running_time=%lf\n", ((double)tend.tv_sec + 1.0e-9*tend.tv_nsec) -
           ((double)tstart.tv_sec + 1.0e-9*tstart.tv_nsec));
  
    printf("Have a nice day!\n");
    return 0;
}

